; Definice globálních proměnných.
globals [
  plastics-count                  ; Úvodní počet plastu v oceánu (ks).
  plastics-weight-sum             ; Úvodní hmotnost plastu v oceánu (kg).
  total-distance-traveled         ; Celková vzdálenost uražená sběrnými loděmi (km).
  total-distance-spent-harvesting ; Vzdálenost uražená při sběru plastu (km).
  total-time-passed               ; Kolik celkového času uběhlo od startu simulace (h).
  total-time-spent-cruising       ; Kolik hodin bylo stráveno běžným cestováním (h).
  total-time-spent-harvesting     ; Kolik hodin bylo stráveno sběrem plastu (h).
  updated-plastics-count          ; Kolik plastu je ještě v oceánu (ks).
  updated-plastics-weight-sum     ; Kolik plastu je ještě v oceánu (kg).
  ocean-cleaned                   ; Zda je v oceánu nějaký plast či nikoliv.
  x-center-plastics               ; X-souřadnice úvodního shluku plastu.
  y-center-plastics               ; Y-souřadnice úvodního shluku plastu.
]

; Definice entit.
breed [plastics plastic]          ; Entita představující kus plastu.
breed [ports port]                ; Entita představující přístav.
breed [boats boat]                ; Entita představující sběrnou loď.

; Definice vlastností plastových částic.
plastics-own [
  weight                          ; Váha jednotlivého plastu.
  plastic-speed                   ; Rychlost pohybu plastu ve vodě.
  is-owned                        ; Pomocná proměnná k určení, zda k plastu míří nějaká sběrná loď. Používá se jen při plavbě z přístavu nebo při otočce, aby dvě lodě nepluly stejným směrem.
]

; Definice vlastností sběrných lodí.
boats-own [
  capacity                       ; Kapacita nákladu. Uvádí se celková kapacita která se sestává z násobků 1000, což je běžná kapacita jednoho námořního kontejneru.
  home-port-location             ; Domovský přístav. Odsud vyplouvá a vrací se vyložit sběrná loď.
  target                         ; Nejbližší kus plastu když se sběrná loď otáčí či vyjíždí z přístavu. Používá se jen při plavbě z přístavu nebo při otočce, aby dvě lodě nepluli stejným směrem.
  is-emptying                    ; Zda je sběrná loď ve stavu vykládání.
  speed                          ; Rychlost pohybu sběrné lodě po oceánu.
  is-harvesting                  ; Zda je sběrná loď ve stavu sběru.
  direction-end                  ; Definice ohraničení manévru otáčky.
]


; Základní konfigurace simulace
to setup
  clear-all                                                                  ; Vyčištění zdrojů pro čistý start další simulace.
  set ocean-cleaned false                                                    ; Definování, že oceán není vyčištěný.
  set x-center-plastics -12                                                  ; X-souřadnice úvodního shluku plastu.
  set y-center-plastics -45                                                  ; Y-souřadnice úvodního shluku plastu.
  setup-ports                                                                ; Volání procedury pro konfiguraci přístavů.
  setup-boats boats-num                                                      ; Volání procedury pro konfiguraci sběrných lodí.
  setup-plastics number-of-plastics x-center-plastics y-center-plastics 40 0 ; Volání procedury pro konfiguraci plastových částic.

; Importování mapy oceánu a pevniny
  ask patches [set pcolor blue]
  import-pcolors "mapaV5.png"
  ask patches with [not shade-of? blue pcolor] [set pcolor blue]
  reset-ticks
end

; Procedura konfigurace a inicializace přístavu.
to setup-ports
  create-ports 3 [
    set shape "circle"
    set color red
    set size 5
  ]

; Nastavování souřadnic jednotlivých přístavů.
  ask port 0 [ setxy 245 -22 ]
  ask port 1 [ setxy 195 34 ]
  ask port 2 [ setxy 182 205 ]
end

; Procedura konfigurace sběrných lodí.
to setup-boats [number]
  let selected-port-location []

; Výběr přístavu dle souřadnic.
  ifelse selected-port = "Los Angeles" [
    set selected-port-location list 245 -22
  ] [
    ifelse selected-port = "San Francisco" [
      set selected-port-location list 195 34
    ] [
      if selected-port = "Victoria" [
        set selected-port-location list 182 205
      ]
    ]
  ]

  let bx item 0 selected-port-location
  let by item 1 selected-port-location

; Inicializace sběrných lodí.
  create-boats number [
    set shape "cleanboat"                                       ; Nastavení tvaru sběrné lodě.
    set color orange                                            ; Nastavení barvy sběrné lodě.
    set size 5                                                  ; Nastavení velikosti sběrné lodě.
    set capacity boat-capacity                                  ; Nastavení kapacity sběrné lodě - očekává vstup.
    set boat-speed-while-harvesting boat-speed-while-harvesting ; Nastavení rychlosti sběrné lodě při sběru - očekává vstup.
    set target nobody                                           ; Úvodní konfigurace - loď aktuálně nikam nemíří
    set is-harvesting false                                     ; Úvodní konfigurace - loď aktuálně není ve stavu sběru.
    set is-emptying false                                       ; Úvodní konfigurace - loď aktuálně není ve stavu vykládání.
    set direction-end -170                                      ; Úvodní konfigurace - definice bodu otočky - otočky se budou dít na y souřadnicích -170 a 170.
    set speed boat-speed                                        ; Nastavení rychlosti sběrné lodě při cestování - očekává vstup.
    set home-port-location selected-port-location               ; Nastavení domovského přístavu sběrné lodě - očekává vstup.
    setxy bx by +(- 1.5 + random-float 3)                       ; Vypuštění sběrných lodí na oceán s lehkou náhodností.
  ]
end


to setup-plastics [number center-x center-y stddev min-radius]
  set plastics-count number                                                      ; Nastavení úvodního počtu plastů - očekává vstup.
  let weight-per-plastic 0.000714041                                             ; Nastavení hmotnosti každého plastu na 0.000714041 kg.

  create-plastics number [
    setxy (random-normal center-x stddev) (random-normal center-y stddev)        ; Nastavení pozice pomocí normální distribuce kolem středu daných souřadnic - umožňuje dojem roztroušení plastů po oceánu.
    set is-owned false                                                           ; Nastavení toho, aby ihned po vygenerování k plastu nemířila nějaká sběrná loď.

    ; Nastavení vizuálů
    set shape "circle"
    let colors random 150
    set size 2

    set plastic-speed 0.01 + random 0.01                                         ; Nastavení pohybu plastu s drobnou náhodností.
    if min-radius > 0 [ rt random 360 fd 100 ]                                   ; Pokud je min-radius větší než 0, plast se pohne, aby se zabránilo přílišnému shlukování uprostřed.

    ; Nastavení hmotnosti plastu
    set weight weight-per-plastic
  ]
end


to go
  plastic-movement-actions                                                       ; Volání procedury pro pohyb plastů.
  plastic-income-actions                                                         ; Volání procedury pro přísun plastů.
  boats-cruise-actions                                                           ; Volání procedury pro běžné cestování sběrné lodě.
  boats-harvesting-actions                                                       ; Volání procedury pro sběrné aktivity sběrné lodě.
  boats-emptying-actions                                                         ; Volání procedury pro vykládání sběrné lodě v přístavu.
  time-passing                                                                   ; Volání procedury pro sčítání času.

; Aktualizace stávajícího počtu plastů a jejich hmotností v oceánu.
; Pokud je zbývající počet plastů menší jak 5, oceán je považován za vyčištěný.
  set updated-plastics-count count plastics
  set updated-plastics-weight-sum sum [weight] of plastics
    if updated-plastics-count < 5 [
    show-metrics
    stop
  ]
  tick
end

; Přičtení jedné hodiny.
to time-passing
  set total-time-passed total-time-passed + 1
end

; Procedura cestování sběrné lodě.
to boats-cruise-actions
  ; Pro každou loď se vypočítá a vykoná pohyb dle aktuálního stavu.
  ask boats [
   ; Pokud nemá sběrná loď nastavený cíl a zrovna nesbírá plast, jde nějaký cíl zaměřit. Loď je v přístavu nebo v otočce.
    if target = nobody and not is-harvesting [
      target-setting
    ]

  ; Pokud sběrná loď má cíl, tak se k němu "natočí". Loď jede k prvnímu plastu.
    if target != nobody [
      face target
    ]

  ; Pokud sběrná loď nemá přímo nastavený cíl a aktuálně je v režimu sběru otočí se po rovnoběžce k bodu manévru.
    if target = nobody and is-harvesting [
      facexy direction-end ycor
    ]

  ; Pokud sběrná loď má plnou sběrnou kapacitu, otočí se směrem k přístavu, odkud vyjela.
    if capacity = 0 [
      facexy item 0 home-port-location item 1 home-port-location
    ]

  ; Tato část kódu zajišťuje manévry sběrné lodě v závislosti na její aktuální souřadnici - v závislosti na tom vypíná sběrný mód.
    if xcor < -170 or 170 < xcor  [
      set is-harvesting false
      if xcor < -170 [
        set direction-end 170
      ]
      if 170 < xcor [
        set direction-end -170
      ]
    ]

    (ifelse
      is-harvesting [                       ; Chování sběrné lodě při sběru - sníží se rychlost na přirovnatelnou k lidské chúzi a přičítá se čas na proměnné počítající čas strávený sběrem plastu a celkový čas.
        if boat-speed-while-harvesting = 1  [fd 0.125 set total-distance-traveled total-distance-traveled + 1  set total-distance-spent-harvesting total-distance-spent-harvesting + 1  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 2  [fd 0.25  set total-distance-traveled total-distance-traveled + 2  set total-distance-spent-harvesting total-distance-spent-harvesting + 2  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 3  [fd 0.375 set total-distance-traveled total-distance-traveled + 3  set total-distance-spent-harvesting total-distance-spent-harvesting + 3  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 4  [fd 0.5   set total-distance-traveled total-distance-traveled + 4  set total-distance-spent-harvesting total-distance-spent-harvesting + 4  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 5  [fd 0.625 set total-distance-traveled total-distance-traveled + 5  set total-distance-spent-harvesting total-distance-spent-harvesting + 5  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 6  [fd 0.75  set total-distance-traveled total-distance-traveled + 6  set total-distance-spent-harvesting total-distance-spent-harvesting + 6  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 7  [fd 0.875 set total-distance-traveled total-distance-traveled + 7  set total-distance-spent-harvesting total-distance-spent-harvesting + 7  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 8  [fd 1     set total-distance-traveled total-distance-traveled + 8  set total-distance-spent-harvesting total-distance-spent-harvesting + 8  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 9  [fd 1.125 set total-distance-traveled total-distance-traveled + 9  set total-distance-spent-harvesting total-distance-spent-harvesting + 9  set total-time-spent-harvesting total-time-spent-harvesting + 1]
        if boat-speed-while-harvesting = 10 [fd 1.25  set total-distance-traveled total-distance-traveled + 10 set total-distance-spent-harvesting total-distance-spent-harvesting + 10 set total-time-spent-harvesting total-time-spent-harvesting + 1]
      ]
      capacity = 0 and not is-emptying [    ; Chování sběrné lodě při vracení se do přístavu -  přičítá se čas na proměnné počítající čas strávený cestováním a celkový čas.
        if boat-speed = 20 [fd 2.50 set total-distance-traveled total-distance-traveled + 20 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 25 [fd 3.13 set total-distance-traveled total-distance-traveled + 25 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 30 [fd 3.75 set total-distance-traveled total-distance-traveled + 30 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 35 [fd 4.38 set total-distance-traveled total-distance-traveled + 35 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 40 [fd 5.00 set total-distance-traveled total-distance-traveled + 40 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 45 [fd 5.63 set total-distance-traveled total-distance-traveled + 45 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 50 [fd 6.25 set total-distance-traveled total-distance-traveled + 50 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 55 [fd 6.88 set total-distance-traveled total-distance-traveled + 55 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 60 [fd 7.50 set total-distance-traveled total-distance-traveled + 60 set total-time-spent-cruising total-time-spent-cruising + 1]
      ]
      not is-emptying [                    ; Chování sběrné lodě při výjezdu z přístavu a manévrech otáčení -  přičítá se čas na proměnné počítající čas strávený cestováním a celkový čas.
        if boat-speed = 20 [fd 2.50 set total-distance-traveled total-distance-traveled + 20 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 25 [fd 3.13 set total-distance-traveled total-distance-traveled + 25 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 30 [fd 3.75 set total-distance-traveled total-distance-traveled + 30 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 35 [fd 4.38 set total-distance-traveled total-distance-traveled + 35 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 40 [fd 5.00 set total-distance-traveled total-distance-traveled + 40 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 45 [fd 5.63 set total-distance-traveled total-distance-traveled + 45 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 50 [fd 6.25 set total-distance-traveled total-distance-traveled + 50 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 55 [fd 6.88 set total-distance-traveled total-distance-traveled + 55 set total-time-spent-cruising total-time-spent-cruising + 1]
        if boat-speed = 60 [fd 7.50 set total-distance-traveled total-distance-traveled + 60 set total-time-spent-cruising total-time-spent-cruising + 1]
      ]
      [
        fd 0
    ])
  ]
end

; Sběrné lodě hledají nejbližší nezaměřený plast v až 1000000000 patchů, a pokud takový najde, zaměří ho.
to target-setting
  if any? plastics in-radius 1000000000 [
    let min-dist 1000000000                                ; Aktuální vzdálenost nejbližšího nezaměřeného cíle sběrné lodě.
    let nearest-plastic nobody                             ; Aktuálně nejbližší nezaměřeny cíl sběrné lodě.
    ask plastics [
      if distance myself < min-dist and is-owned = false [ ; Pokud najde bližší plast a nezaměřený, provede se následující kód.

      ; Aktualizuje se minimální vzdálenost od plastu na vzdálenost od nalezeného nejbližšího plastu
        set min-dist distance myself

      ; Původní nejbližší plast již není zaměřený
        if nearest-plastic != nobody [
          ask nearest-plastic [
            set is-owned false
          ]
        ]
      ; Aktualizuje se nejbližší plast na nalezený nejbližší plast a nastaví ho na zaměřený
        set nearest-plastic self
        set is-owned true
      ]
    ]
    set target nearest-plastic
  ]
end

; Sběrné lodě sbírají plasty v moři
to boats-harvesting-actions

  ask boats [
    ; Pokud sběrná loď našla cíl a vzdálenost k cíli je menší nebo rovna její samotné velikosti, zapne se sběrný mód.
    if target != nobody and distance target <= size [
      set is-harvesting true
    ]
    if is-harvesting [
      set speed boat-speed-while-harvesting                        ; Ve chvíli kdy sběrná loď sbírá, nastaví se jí patřičná, velmi snížená, rychlost.
      let h-boat self                                              ; Nastaví sama sebe na dočasnou proměnnou, díky které zadává další příkazy.

    ; Každý plast v radiusu velikosti lodi se pokudí loď sebrat
      ask plastics in-radius size [
        let real-boat-fill min (list [capacity] of h-boat weight)  ; Detekce možnosti sběru. Pokud loď má větší kapacitu než je objem plastu, tak ho sebere. V opačném případě loď doplní se do plna a zbytek plastu zůstane ve vodě.

        set weight weight - real-boat-fill                         ; Abstrakce sběru plastu - plastu se ubere jeho individuální hmotnost právě o tolik, kolik se naplnila kapacita sběrné lodě.

        ask h-boat [

        ; Snížení kapacity sběrné lodě právě o tolik, kolik bylo sebráno sebranému plastu z jeho individuální hmotnosti.
        ; Ve chvíli kdy tato kapacita klesne na 0, ruší se mód sběru.
          set capacity capacity - real-boat-fill
          if capacity = 0 [
            set is-harvesting false
          ]
        ]

      ; Ve chvíli kdy individuální hmotnost plastu dosáhne nuly, znamená to, že byl sebrán, a přestává na mapě existovat.
        if weight = 0 [
          die
        ]

     ; Ve chvíli, kdy je kapacita sběrné lodě na nule, končí procedura pro tuto loď, jelikož se musí vrátit do přístavu a vyložit náklad.
        if [capacity] of h-boat = 0 [
          stop
        ]
      ]
    ]
  ]
end

to boats-emptying-actions
  ask boats [
  ; Pokud je sběrná kapacita sběrné lodě na nule a zrovna nesbírá a nevykládá, podívá se, zda je v blízkosti přístav - pokud ano, jde vykládat, a nastavuje vykládací režim.
    if capacity = 0 and not is-harvesting and not is-emptying [
      if any? ports in-radius 6 [ ; Detekce přístavu v radiusu 6 - ochraná konstanta pro větší rychlosti.
        set is-emptying true
      ]
    ]

  ; Pokud je sběrná loď ve stavu vykládání, podívá se, zda je její kapacita přičtená o rychlost vykládání menší než celková kapacita lodě.
  ; Pokud ano, kapacita se začne uvolňovat a loď již vykládá.
    if is-emptying [
      ifelse (capacity + emptying-speed) < boat-capacity [
        set capacity capacity + emptying-speed
      ] [
      ; Ve chvíli kdy je vyloženo, přičtou se potřebné časy pro kontrolu a takový čas se zdvojnásobí pro pokrytí jak vyložení, tak i naložení, které je považováno za stejnou časovou délku.
        set capacity boat-capacity                   ; Obnovení sběrné kapacity sběrné lodě na její maximum.
        set target nobody                            ; Vynulování cíle sběrné lodě.
        set is-emptying false                        ; Rušení režimu vykládání.
        set direction-end abs direction-end * -1     ; Sběrná loď se otočí směrem od přístavu.
      ]
    ]
  ]
end

; Procedura pro pohyb plastu z důvodu projevu oceánských proudů.
to plastic-movement-actions
  ; Pro každý plast se vykoná pohyb, který je přibližným dostředným pohybem po kružnici kolem definovaného středu
  ask plastics [
    facexy x-center-plastics y-center-plastics                 ; Nejdříve se plast otočí ke středu
    fd distancexy x-center-plastics y-center-plastics / 100000 ; Posun plastu směrem ke středu plastového shluku, tak aby se pohyb s kratší vzdáleností ke středu zpomalu zpomaloval. Lineární zpomalení s experimentálně zvolenou konstantou 1/100000.
    lt 89                                                      ; Otočí plast o určité stupně doleva. Hodnota 89 stupňů je zvolena experimentálne pro udržení dostředného pohybu.
    fd plastic-speed                                           ; Posun dle definované rychlosti.
  ]
end

; Procedura přísunu plastu. Využívá se stejný funkce jako pro iniciaci, ale je definovaný radius v kterém se nové plasty budou objevovat
to plastic-income-actions
  setup-plastics plastic-income x-center-plastics y-center-plastics 10 160
end



; Ve chvíli kdy je tato procedura zavolána, nastavuje se cílová proměnná jako pravda, a je vyslána zpráva, že je oceán vyčištěn.
to show-metrics
  set ocean-cleaned true
  user-message (word "Oceán vyčištěn!")
end
@#$#@#$#@
GRAPHICS-WINDOW
579
10
1301
733
-1
-1
1.42515
1
10
1
1
1
0
0
0
1
-250
250
-250
250
0
0
1
ticks
60.0

BUTTON
0
10
237
43
NASTAVIT SIMULACI DLE ZADANÝCH HODNOT
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
0
46
236
79
SPUSTIT SIMULACI
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
0
112
237
145
boats-num
boats-num
1
10
10.0
1
1
lodí
HORIZONTAL

MONITOR
1307
10
1519
71
Aktuální počet plastu v oceánu
updated-plastics-count
0
1
15

SLIDER
277
281
574
314
boat-capacity
boat-capacity
8000
20000
8000.0
1000
1
kg
HORIZONTAL

MONITOR
1523
10
1765
71
Kolik kg plastu je v oceánu
updated-plastics-weight-sum
0
1
15

SLIDER
278
381
575
414
emptying-speed
emptying-speed
6000
20000
6000.0
1000
1
Kg/h
HORIZONTAL

MONITOR
1305
114
1765
159
Celkem uraženo kilometrů
total-distance-traveled
0
1
11

SLIDER
0
235
236
268
boat-speed
boat-speed
20
60
45.0
5
1
km/h
HORIZONTAL

MONITOR
1520
346
1764
391
Uplynulý čas [hodiny]
total-time-passed
0
1
11

MONITOR
1305
346
1518
391
Uplynulý čas [dny]
total-time-passed / 24
0
1
11

CHOOSER
0
166
235
211
selected-port
selected-port
"Victoria" "San Francisco" "Los Angeles"
0

MONITOR
277
316
574
361
Počet kontejnerů s kapacitou 1000 kg
boat-capacity / 1000
0
1
11

SLIDER
1
353
235
386
boat-speed-while-harvesting
boat-speed-while-harvesting
5
10
10.0
1
1
km/h
HORIZONTAL

TEXTBOX
1
333
216
351
Rychlost lodě při sběru plastu
15
0.0
1

TEXTBOX
0
91
150
110
Počet lodí
15
0.0
1

TEXTBOX
277
261
427
280
Kapacita lodě
15
0.0
1

TEXTBOX
0
216
272
234
Rychlost lodě při běžném cestování
15
0.0
1

MONITOR
0
270
236
315
Rychlost v uzlech
boat-speed / 1.82
0
1
11

MONITOR
1
388
236
433
Rychlost v uzlech
boat-speed-while-harvesting / 1.82
0
1
11

TEXTBOX
278
362
514
380
Rychlost vykládky plastu v přístavu
15
0.0
1

TEXTBOX
0
147
150
166
Přístav
15
0.0
1

TEXTBOX
274
91
715
109
Počet původního množství plastů v oceánu
15
0.0
1

TEXTBOX
281
628
574
646
Odhad frekvence přísunu plastu za hodinu
15
0.0
1

TEXTBOX
275
16
551
92
Realita: \nPočet: 108 000 000 000 kusů plastu\nCelková hmotnost: 77 116 393 kg
15
0.0
1

MONITOR
275
216
574
261
Měřítko [1:X]
108000000000 / number-of-plastics
0
1
11

MONITOR
1306
395
1518
440
Čas strávený sběrem [h]
total-time-spent-harvesting / boats-num
0
1
11

MONITOR
1521
394
1764
439
Čas strávený cestováním [h]
total-time-spent-cruising / boats-num
0
1
11

MONITOR
1307
443
1518
488
Čas strávený sběrem [%]
(total-time-spent-harvesting / boats-num) / total-time-passed * 100
0
1
11

MONITOR
1522
442
1764
487
Čas strávený cestováním [%]
(total-time-spent-cruising / boats-num) / total-time-passed * 100
0
1
11

MONITOR
1304
297
1763
342
Uplynulý čas [měsíce]
total-time-passed / 720
0
1
11

MONITOR
1306
162
1518
207
Celkem uraženo kilometrů sběrem
total-distance-spent-harvesting
0
1
11

MONITOR
1520
162
1764
207
Celkem uraženo kilometrů cestováním
total-distance-traveled - total-distance-spent-harvesting
0
1
11

MONITOR
1305
210
1519
255
Podíl sběrné vzdálenosti k celku [%]
(total-distance-spent-harvesting / total-distance-traveled) * 100
0
1
11

MONITOR
1522
209
1764
254
Podíl cestovací vzdálenosti k celku [%]
((total-distance-traveled - total-distance-spent-harvesting) / total-distance-traveled) * 100
0
1
11

TEXTBOX
1307
90
1756
109
Vzdálenostní dimenze [za všechny lodě]
15
0.0
1

TEXTBOX
1308
271
1550
309
Časová dimenze [konstantní]
15
0.0
1

PLOT
1309
491
1765
611
Pokles počtu plastu v čase
Čas
Počet
0.0
1.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -13840069 true "" "plot updated-plastics-count"

SLIDER
274
178
573
211
number-of-plastics
number-of-plastics
1080
108000000000
10800.0
1080
1
NIL
HORIZONTAL

INPUTBOX
274
114
574
174
number-of-plastics
10800.0
1
0
Number

TEXTBOX
548
140
571
162
*
30
15.0
1

TEXTBOX
6
446
255
668
* Pozor na hodnoty > 200 000 - pokud nemáte výkonný počítač, NetLogo výpočetní složitost nezvládne. Slidery jsou pouze pro ilustraci a umožnění pochopení o jak velká čísla se v realitě jedná. Pro samotné testování využijte vstupní políčka a manuálně zadejte rozumné hodnoty. Též můžete vidět měřítko symbolizující porovnání shluku v simulaci a v realitě.
15
15.0
1

MONITOR
439
648
569
693
Počet plastů za den [ks]
plastic-income * 24
0
1
11

MONITOR
439
695
569
740
Přísun plastu za den [kg]
plastic-income * 24 * 0.000714041
0
1
11

BUTTON
280
555
574
588
Nastavit hodnoty na měřítko 1 : 1 000 000 k realitě
set number-of-plastics 108000
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
280
519
575
552
Nastavit hodnoty na měřítko 1 : 10 000 000 k realitě
set number-of-plastics 10800
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
278
419
575
478
Nastavit hodnoty na normu [vždy před nastavením měřítek]
set number-of-plastics 0\nset boat-capacity 8000\nset emptying-speed 6000
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
279
482
575
515
Nastavit hodnoty na měřítko 1 : 100 000 000 k realitě
set number-of-plastics 1080
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
281
592
575
625
Nastavit hodnoty na měřítko 1 : 1 k realitě [POZOR]
set number-of-plastics 108000000000\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
280
647
435
740
plastic-income
0.0
1
0
Number

TEXTBOX
258
438
278
463
1.
20
0.0
1

TEXTBOX
256
487
288
516
a)\n
20
0.0
1

TEXTBOX
255
522
284
548
b)
20
0.0
1

TEXTBOX
257
556
280
582
c)
20
0.0
1

TEXTBOX
257
592
284
617
d)
20
15.0
1

PLOT
1310
614
1766
734
Sběrné kilometry [zeleně] / Cestované kilometry [červeně]
Čas
Kilometry
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"Cestované kilometry" 1.0 0 -2674135 true "" "plot total-distance-traveled - total-distance-spent-harvesting"
"Sběrné kilometry" 1.0 0 -10899396 true "" "plot total-distance-spent-harvesting / boats-num"

@#$#@#$#@
Definice problému

Vynálezem plastu člověk zhotovil zcela nový problém, který tkví v samotné podstatě plastových výrobků - jejich vlastnosti, které jsou pro konzumní společnost tak lákavé, jsou zároveň velkým trnem v oku přírody, jelikož rozklad takového plastu trvá neprakticky dlouho. Jedním z dopadů neschopnosti lidstva bezpečně a spolehlivě zpracovávat plastové odpady na makroekonomickém měřítku je znečištění řek a oceánů. Tato práce se pokouší pomocí simulace takového znečištění přinést užitečný vhled do problematiky jak pro laika, který nemá o základních skutečnostech povědomí, tak pro zkušené experty, kteří mohou potenciálně výsledky simulace a spojitosti v ní vytvořené použít pro podporu rozhodovacího procesu v rámci definice klíčových faktorů nejen na samotné znečištění oceánů, ale i na jeho mitigaci. Jednou z organizací, která si dala za úkol tento problém řešit, je Ocean Cleanup , z jejichž přístupu a metod tato simulace vychází. Oblast zaměření simulace je orientována na největší shluk plastu v oceánu, kterým je Great Pacific Garbage Patch (GPGP) , který se nachází západně od Kalifornie, USA.

Cílem simulace je zkoumat mitigační schopnosti úklidových lodí v oblasti Great Pacific Garbage Patch a navrhnout hrubý plán projektu, který by symbolizoval optimální počet nasazených úklidových lodí pro efektivní odbyt plastu a hlavní faktory příspívající do navyšujícího se znečištění a efektivity samotného úklidu.

Poznámka: Autor před začátkem projektu - tedy při vypracování projektového záměru - nebyl schopen přesně odhadnout technické překážky či proveditelnost / reference na určité aspekty projektového záměru, čili ačkoliv je záměr projektu a jeho oblast totožná, jedná se o úpravy ve formě proměnných, což však vzhledem k velké šíře náhodných vlivů v realitě nemá příliš velký dopad na účel simulace, která je tak či tak pouze odhadem a pomůckou. 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cleanboat
true
0
Polygon -2674135 true false 150 300 0 0 15 0 150 270 285 0 300 0 150 300

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
